/*
 * Copyright (C) 2012-2016 Protein Metrics Inc. - All Rights Reserved.
 * Unauthorized copying or distribution of this file, via any medium is strictly prohibited.
 * Confidential.
 */
/** Module Design Pattern
 Internal module used as validator for JS object */
var validateInputObject = (function () {

    function isUndefined(value) {
        if (value === undefined) {
            return true;
        }

        return false;
    }

    function basicObjectValidate(obj) {
        if (isUndefined(obj) || obj === null) {
            return false;
        }
        return true;
    }

    return {
        isValidNumberObject: function(obj) {
            return basicObjectValidate(obj) && !isNaN(obj);
        },
        isValidStringObject: function(obj) {
            return basicObjectValidate(obj) && obj.length > 0;
        },
        isUndefined: isUndefined
    }

})();

/*Internal module used as helper object for join function */
var joinFunctionCreator = (function () {
    var countOfValues = function(values) {
        var max = 0;

        for (var i = 0; i < values.length; ++i) {
            var length = values[i].length;
            if (max < length) {
                max = length;
            }
        }

        return max;
    };

    /** If value exist return value if not return first which is defined */
    var detectValue = function(listsOfValues, j, i) {
        var value = listsOfValues[j][i];

        if (value !== undefined) {
            return value;
        }

        for (var index = 0; index < countOfValues(listsOfValues); ++index) {
            if (listsOfValues[j][index] !== undefined) {
                return listsOfValues[j][index];
            }
        }

        return value;
    };

    /** Returns true if all of the values are missing (undefined or empty), otherwise false is returned */
    var isEmpty = function(listsOfValues) {
        var count = countOfValues(listsOfValues);
        for (var j = 0; j < listsOfValues.length; ++j) {
            for (var index = 0; index < count; ++index) {
                if (listsOfValues[j][index]) {
                    return false;
                }
            }
        }
        return true;
    };

    return {
        countOfValues: countOfValues,
        detectValue: detectValue,
        isEmpty: isEmpty
    }

})();

/** Module used to convert undefined object to empty string.
 One of use case is to avoid get undefined object string in template editor content */
var dataTypesConverter = (function () {

    function fixUpObject(value) {
        if (validateInputObject.isUndefined(value)) {
            return '';
        }
        return value;
    }

    return {
        fixUpObject: fixUpObject
    }

})();

/**Module used to provide some basic skelton for Js functions */
var templateFunctionsManager = (function () {

    var stringReturnTemplateWithErrorHandling = function(valueToReturnIfSuccess) {
        if (!validateInputObject.isValidStringObject(valueToReturnIfSuccess)) {
            return dataTypesConverter.fixUpObject(valueToReturnIfSuccess);
        }

        return valueToReturnIfSuccess;
    };

    var getValue = function(element) {
        if (typeof element === "object") {

            /** Find first not empty value */
            for (var i = 0; i < element.length; ++i) {
                if (element[i].length > 0) {
                    return element[i];
                }
            }
        }
        return element;
    };

    var detectValue = function(inputValue, convertToNumber) {
        var inputParametersValue = inputParameters[inputValue];

        if (inputParametersValue !== undefined) {
            if (inputParametersValue.length === 0) {
                return inputParametersValue;
            }
            var detectedValue = getValue(inputParametersValue);

            if (convertToNumber ? !detectedValue : detectedValue.trim().length === 0) {
                return inputValue;
            }
            inputValue = String(inputValue).split(inputValue).join(convertToNumber ? Number(detectedValue) : detectedValue);
        }
        return inputValue;
    };

    return {
        stringReturnTemplateWithErrorHandling: stringReturnTemplateWithErrorHandling,
        detectValue: detectValue
    }

})();

/** More accurate way to round with decimals */
var round = function(value, decimals) {

    value = templateFunctionsManager.detectValue(value, true);

    if (!validateInputObject.isValidNumberObject(value)) {
        return dataTypesConverter.fixUpObject(value);
    }

    if (decimals === undefined || decimals === null) {
        decimals = 2;
    }

    var returnedValue = Number(Math.round(value + 'e' + decimals) + 'e-' + decimals);
    if (isNaN(returnedValue)) {
        return Math.round(value);
    }
    else {
        return returnedValue;
    }
};

var firstLetter = function(str) {
    str = templateFunctionsManager.detectValue(str);
    return templateFunctionsManager.stringReturnTemplateWithErrorHandling(str[0]);
};

var upper = function(str) {
    str = templateFunctionsManager.detectValue(str);
    return templateFunctionsManager.stringReturnTemplateWithErrorHandling(str.toUpperCase());
};

var join = function(template, separator, skipEmptyParts, hideSeparatorForEmptyEntries) {
    var returnedValue = [];

    if (separator === undefined) {
        separator = '; ';
    }
    
    if (skipEmptyParts === undefined) {
        skipEmptyParts = true;
    }

    if (hideSeparatorForEmptyEntries === undefined) {
        hideSeparatorForEmptyEntries = false;
    }

    if (!inputParameters) {
        return template;
    }

    var listsOfValues = [];
    /** list of used keys */
    var keys = [];
    Object.keys(inputParameters).forEach(function(key,index) {
        var values = inputParameters[key];

        if (template.indexOf(key) === -1) {
            return;
        }
        if (typeof values !== "object") {
            values = values.split();
        }
        listsOfValues.push(values);
        keys.push(key);
    });

    if (listsOfValues.length === 0) {
        return template;
    }

    if (skipEmptyParts === true && joinFunctionCreator.isEmpty(listsOfValues) === true) {
        return "";
    }

    var countOfValues = joinFunctionCreator.countOfValues(listsOfValues);
    for (var i = 0; i < countOfValues; ++i) {

        for (var j = 0; j < listsOfValues.length; ++j) {
            if (returnedValue[i] === undefined) {
                returnedValue[i] = template;
            }

            returnedValue[i] = returnedValue[i].split(keys[j]).join(joinFunctionCreator.detectValue(listsOfValues, j, i));
        }
    }

    if (hideSeparatorForEmptyEntries === true) {
        returnedValue = returnedValue.filter(function (el) {
            if (typeof el === "string") {
                return el.length > 0;
            }
            return typeof el !== 'undefined' && el !== null && !isNaN(el);
        });
    }

    return returnedValue.join(separator);
};

var max = function (column_of_interest, column_proxy) {
    if (column_of_interest === undefined) {
        return "";
    }
    if (column_proxy === undefined || column_proxy === null) {
        column_proxy = column_of_interest;
    }

    // using Object.keys is slightly more robust than something like inputParameters[column_of_interest] in case there is user error in column_of_interest
    var listOfInterest = [];
    var listProxy = [];
    Object.keys(inputParameters).forEach(function (key, index) {
        var values = inputParameters[key]; // values is a single object containing all data for one column
        if (column_of_interest.indexOf(key) !== -1) {
            listOfInterest.push(values.slice());
        }
        if (column_proxy.indexOf(key) !== -1) {
            listProxy.push(values.slice());
        }
    });
    if (listOfInterest.length === 0 || listProxy.length === 0) {
        return "";
    }

    listOfInterest = listOfInterest[0]; // take the max of the first column in column_of_interest (might be more than one column if user error)
    var maxNum = Number.NEGATIVE_INFINITY;
    var maxIdx = [-1];
    for (var i = 0, size = listOfInterest.length; i < size; i++) {
        listOfInterest[i] = String(listOfInterest[i]).split("%")[0]; // for columns with % data
        if (listOfInterest[i] === "") { // empty value
            continue;
        }
        listOfInterest[i] = Number(listOfInterest[i]);
        if (Number.isNaN(listOfInterest[i])) { // string data
            continue;
        }
        if (listOfInterest[i] > maxNum) {
            maxNum = listOfInterest[i];
            maxIdx = [i];
        }
        else if (listOfInterest[i] === maxNum) {
            maxIdx.push(i);
        }
    }

    var ret = [];
    for (var i = 0, size = listProxy.length; i < size; i++) { // support option for multiple column proxy
        var listProxyColumn = listProxy[i];
        if (maxIdx[0] === -1) {
            ret.concat(listProxyColumn); // if column_of_interest is all empty values or string values, add all values in column_proxy
            continue;
        }
        for (j = 0; j < maxIdx.length; j++) {
            ret.push(listProxyColumn[maxIdx[j]]); // add the maximum(s) for the current column in column_proxy
        }
    }
    return ret;
};


/** To test substring catch */
var supper = function(str) {
    return 'supper function called!';
};

var lower = function(str) {
    str = templateFunctionsManager.detectValue(str);
    return templateFunctionsManager.stringReturnTemplateWithErrorHandling(str.toLowerCase());
};